# airline_functions.py
"""
Simulated airline booking system functions.
In a real system, these would connect to actual databases and APIs.
"""

# Simulated database of bookings
bookings_db = {
    "ABC123": {
        "booking_id": "ABC123",
        "passenger_name": "John Doe",
        "flight_number": "AA101",
        "route": "New York (JFK) to London (LHR)",
        "date": "2026-06-15",
        "seat": "14B",
        "status": "confirmed",
        "email": "john.doe@email.com"
    },
    "XYZ789": {
        "booking_id": "XYZ789",
        "passenger_name": "Jane Smith",
        "flight_number": "AA202",
        "route": "Los Angeles (LAX) to Tokyo (NRT)",
        "date": "2026-07-20",
        "seat": "23A",
        "status": "confirmed",
        "email": "jane.smith@email.com"
    }
}


def check_booking(confirmation_number: str) -> dict:
    """
    Look up flight booking details by confirmation number.
    
    Args:
        confirmation_number: The booking confirmation number
        
    Returns:
        Dictionary with booking details or error message
    """
    print(f"[FUNCTION CALLED] check_booking('{confirmation_number}')")
    
    if confirmation_number in bookings_db:
        booking = bookings_db[confirmation_number]
        return {
            "success": True,
            "booking": booking
        }
    else:
        return {
            "success": False,
            "error": f"No booking found with confirmation number {confirmation_number}"
        }


def modify_seat(booking_id: str, new_seat: str) -> dict:
    """
    Change the seat assignment for a booking.
    
    Args:
        booking_id: The booking confirmation number
        new_seat: The new seat number (e.g., "12A")
        
    Returns:
        Dictionary with success status and message
    """
    print(f"[FUNCTION CALLED] modify_seat('{booking_id}', '{new_seat}')")
    
    if booking_id not in bookings_db:
        return {
            "success": False,
            "error": f"No booking found with ID {booking_id}"
        }
    
    # Validate seat format (basic validation)
    if not new_seat or len(new_seat) < 2:
        return {
            "success": False,
            "error": "Invalid seat number format"
        }
    
    # Update the seat
    old_seat = bookings_db[booking_id]["seat"]
    bookings_db[booking_id]["seat"] = new_seat
    
    return {
        "success": True,
        "message": f"Seat changed from {old_seat} to {new_seat}",
        "new_seat": new_seat
    }


def process_refund(booking_id: str, amount: float) -> dict:
    """
    Process a refund for a booking cancellation.
    
    Args:
        booking_id: The booking confirmation number
        amount: The refund amount in USD
        
    Returns:
        Dictionary with refund confirmation
    """
    print(f"[FUNCTION CALLED] process_refund('{booking_id}', {amount})")
    
    if booking_id not in bookings_db:
        return {
            "success": False,
            "error": f"No booking found with ID {booking_id}"
        }
    
    # Update booking status
    bookings_db[booking_id]["status"] = "refunded"
    
    return {
        "success": True,
        "message": f"Refund of ${amount:.2f} processed successfully",
        "refund_amount": amount,
        "booking_id": booking_id
    }


def send_email(customer_email: str, message: str) -> dict:
    """
    Send an email to the customer.
    
    Args:
        customer_email: Customer's email address
        message: Email message content
        
    Returns:
        Dictionary with send confirmation
    """
    print(f"[FUNCTION CALLED] send_email('{customer_email}', message_length={len(message)})")
    
    # In a real system, this would integrate with an email service
    # For now, we just simulate it
    
    print(f"  EMAIL TO: {customer_email}")
    print(f"  SUBJECT: Booking Update")
    print(f"  MESSAGE: {message}")
    
    return {
        "success": True,
        "message": f"Email sent to {customer_email}",
        "email": customer_email
    }

# Mapping of function names to actual Python functions
available_functions = {
    "check_booking": check_booking,
    "modify_seat": modify_seat,
    "process_refund": process_refund,
    "send_email": send_email
}

