import os
import time
import json
from azure.ai.agents import AgentsClient
from azure.ai.agents.models import ListSortOrder, FunctionTool, ToolSet, RequiredAction
from azure.identity import DefaultAzureCredential
from dotenv import load_dotenv

# Import our airline functions
from airline_functions import available_functions, check_booking, modify_seat, process_refund, send_email

# Load environment variables
load_dotenv()

# Get configuration
endpoint = os.getenv("AZURE_AI_ENDPOINT")
model_deployment = os.getenv("MODEL_DEPLOYMENT_NAME")

# Validate configuration
if not all([endpoint, model_deployment]):
    raise ValueError("Missing required environment variables")

# Create AgentsClient
agents_client = AgentsClient(
    endpoint=endpoint,
    credential=DefaultAzureCredential(exclude_environment_credential=True)
)

print("Connected to Azure AI Foundry")

# Create FunctionTool with actual Python functions
functions = FunctionTool(functions=[check_booking, modify_seat, process_refund, send_email])

# Create agent with function calling capability
with agents_client:
    agent = agents_client.create_agent(
        model=model_deployment,
        name="airline-booking-agent",
        instructions="""You are a helpful airline customer service agent. 
        You can help customers with their flight bookings by:
        - Looking up booking details
        - Changing seat assignments
        - Processing refunds
        - Sending confirmation emails
        
        Always be polite and professional. When you successfully complete an action,
        confirm what you did for the customer.""",
        tools=functions.definitions
    )
    print(f"Created agent: {agent.id}")

    # Create thread
    thread = agents_client.threads.create()
    print(f"Created thread: {thread.id}")

    # Customer request
    user_message = "Hi, can you check my booking ABC123 and then change my seat to 12A?"
    
    message = agents_client.messages.create(
        thread_id=thread.id,
        role="user",
        content=user_message
    )
    print(f"\nUser: {user_message}")

    # Run agent
    run = agents_client.runs.create(
        thread_id=thread.id,
        agent_id=agent.id
    )

    # Poll and handle function calls
    while run.status in ["queued", "in_progress", "requires_action"]:
        time.sleep(1)
        run = agents_client.runs.get(thread_id=thread.id, run_id=run.id)
        
        print(f"Run status: {run.status}")
        
        # Handle function calling
        if run.status == "requires_action":
            print("\n[AGENT REQUESTING FUNCTION CALLS]")
            
            # Get the required actions
            tool_calls = run.required_action.submit_tool_outputs.tool_calls
            tool_outputs = []
            
            for tool_call in tool_calls:
                function_name = tool_call.function.name
                function_args = json.loads(tool_call.function.arguments)
                
                print(f"\nFunction: {function_name}")
                print(f"Arguments: {function_args}")
                
                # Execute the function
                if function_name in available_functions:
                    function_to_call = available_functions[function_name]
                    function_response = function_to_call(**function_args)
                    
                    print(f"Response: {function_response}")
                    
                    # Add the function output
                    tool_outputs.append({
                        "tool_call_id": tool_call.id,
                        "output": json.dumps(function_response)
                    })
                else:
                    print(f"Unknown function: {function_name}")
            
            # Submit all function outputs back to the agent
            if tool_outputs:
                print("\n[SUBMITTING FUNCTION OUTPUTS TO AGENT]")
                run = agents_client.runs.submit_tool_outputs(
                    thread_id=thread.id,
                    run_id=run.id,
                    tool_outputs=tool_outputs
                )

    print(f"\nRun completed with status: {run.status}")

    # Check for errors
    if run.status == "failed":
        print(f"Run failed: {run.last_error}")

    # Get messages
    messages = agents_client.messages.list(
        thread_id=thread.id,
        order=ListSortOrder.ASCENDING
    )

    # Display conversation
    print("\n" + "="*60)
    print("CONVERSATION")
    print("="*60)
    for msg in messages:
        if msg.text_messages:
            last_text = msg.text_messages[-1]
            print(f"\n{msg.role.upper()}: {last_text.text.value}")
    print("="*60)